import datetime
from collections.abc import Mapping
from typing import TYPE_CHECKING, Any, TypeVar, Union
from uuid import UUID

from attrs import define as _attrs_define
from attrs import field as _attrs_field
from dateutil.parser import isoparse

from ..types import UNSET, Unset

if TYPE_CHECKING:
    from ..models.item_type import ItemType


T = TypeVar("T", bound="Valuation")


@_attrs_define
class Valuation:
    """
    Example:
        [{'id': '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'valuationType': {'id': '497f6eca-6276-4993-bfeb-53cbbbba6f08',
            'code': 'string', 'name': 'string', 'description': 'string', 'groupMemberOf': [{'id':
            '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code': 'string', 'name': 'string', 'description': 'string'}]},
            'valuationValue': 0, 'valuationIsoCurrency': 'CHF', 'valuationDate': '2019-08-24', 'provider': 'string'}]

    Attributes:
        id (Union[Unset, UUID]):
        valuation_type (Union[Unset, ItemType]):
        valuation_value (Union[Unset, float]): The value of the valuation. Example: Valuation of the House (Building) is
            CHF 1'200'000. The value here is 1200000.
        valuation_iso_currency (Union[Unset, str]): The currency in ISO format of the valuation. Example: Valuation of
            the House (Building) is CHF 1'200'000. The value her is CHF. See: https://de.wikipedia.org/wiki/ISO_4217
            Example: ['CHF'].
        valuation_date (Union[Unset, datetime.date]): Date when the valuation was caluclated.
        provider (Union[Unset, str]): Text for information who the valuation value has calculated. Alternative: You can
            define for each provider an own valuationType and let this field empty. Example: WP-Market-Valuation and IAZI-
            Market-Valuation etc.
    """

    id: Union[Unset, UUID] = UNSET
    valuation_type: Union[Unset, "ItemType"] = UNSET
    valuation_value: Union[Unset, float] = UNSET
    valuation_iso_currency: Union[Unset, str] = UNSET
    valuation_date: Union[Unset, datetime.date] = UNSET
    provider: Union[Unset, str] = UNSET
    additional_properties: dict[str, Any] = _attrs_field(init=False, factory=dict)

    def to_dict(self) -> dict[str, Any]:
        id: Union[Unset, str] = UNSET
        if not isinstance(self.id, Unset):
            id = str(self.id)

        valuation_type: Union[Unset, dict[str, Any]] = UNSET
        if not isinstance(self.valuation_type, Unset):
            valuation_type = self.valuation_type.to_dict()

        valuation_value = self.valuation_value

        valuation_iso_currency = self.valuation_iso_currency

        valuation_date: Union[Unset, str] = UNSET
        if not isinstance(self.valuation_date, Unset):
            valuation_date = self.valuation_date.isoformat()

        provider = self.provider

        field_dict: dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update({})
        if id is not UNSET:
            field_dict["id"] = id
        if valuation_type is not UNSET:
            field_dict["valuationType"] = valuation_type
        if valuation_value is not UNSET:
            field_dict["valuationValue"] = valuation_value
        if valuation_iso_currency is not UNSET:
            field_dict["valuationIsoCurrency"] = valuation_iso_currency
        if valuation_date is not UNSET:
            field_dict["valuationDate"] = valuation_date
        if provider is not UNSET:
            field_dict["provider"] = provider

        return field_dict

    @classmethod
    def from_dict(cls: type[T], src_dict: Mapping[str, Any]) -> T:
        from ..models.item_type import ItemType

        d = dict(src_dict)
        _id = d.pop("id", UNSET)
        id: Union[Unset, UUID]
        if isinstance(_id, Unset):
            id = UNSET
        else:
            id = UUID(_id)

        _valuation_type = d.pop("valuationType", UNSET)
        valuation_type: Union[Unset, ItemType]
        if isinstance(_valuation_type, Unset):
            valuation_type = UNSET
        else:
            valuation_type = ItemType.from_dict(_valuation_type)

        valuation_value = d.pop("valuationValue", UNSET)

        valuation_iso_currency = d.pop("valuationIsoCurrency", UNSET)

        _valuation_date = d.pop("valuationDate", UNSET)
        valuation_date: Union[Unset, datetime.date]
        if isinstance(_valuation_date, Unset):
            valuation_date = UNSET
        else:
            valuation_date = isoparse(_valuation_date).date()

        provider = d.pop("provider", UNSET)

        valuation = cls(
            id=id,
            valuation_type=valuation_type,
            valuation_value=valuation_value,
            valuation_iso_currency=valuation_iso_currency,
            valuation_date=valuation_date,
            provider=provider,
        )

        valuation.additional_properties = d
        return valuation

    @property
    def additional_keys(self) -> list[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties
