import datetime
from collections.abc import Mapping
from typing import TYPE_CHECKING, Any, TypeVar
from uuid import UUID

from attrs import define as _attrs_define
from attrs import field as _attrs_field
from dateutil.parser import isoparse

if TYPE_CHECKING:
    from ..models.upload_description_upload_request_headers import UploadDescriptionUploadRequestHeaders


T = TypeVar("T", bound="UploadDescription")


@_attrs_define
class UploadDescription:
    """All information about the file upload

    Attributes:
        id (UUID):  Example: 158fb228-d101-448f-854d-2101de4d09a6.
        file_name (str):  Example: readme.txt.
        mime_type (str):  Example: text/plain.
        created_by (str):
        put_url (str): The url where the binaries have to be put to. Required headers are: Content-Type
        put_url_valid_until (datetime.datetime): Timestamp until when the putUrl is working. If no binaries are uploaded
            up to this timestamp the file metadata is deleted and a new POST has to be done to initialize a new upload.
        upload_request_headers (UploadDescriptionUploadRequestHeaders):
    """

    id: UUID
    file_name: str
    mime_type: str
    created_by: str
    put_url: str
    put_url_valid_until: datetime.datetime
    upload_request_headers: "UploadDescriptionUploadRequestHeaders"
    additional_properties: dict[str, Any] = _attrs_field(init=False, factory=dict)

    def to_dict(self) -> dict[str, Any]:
        id = str(self.id)

        file_name = self.file_name

        mime_type = self.mime_type

        created_by = self.created_by

        put_url = self.put_url

        put_url_valid_until = self.put_url_valid_until.isoformat()

        upload_request_headers = self.upload_request_headers.to_dict()

        field_dict: dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "id": id,
                "fileName": file_name,
                "mimeType": mime_type,
                "createdBy": created_by,
                "putUrl": put_url,
                "putUrlValidUntil": put_url_valid_until,
                "uploadRequestHeaders": upload_request_headers,
            }
        )

        return field_dict

    @classmethod
    def from_dict(cls: type[T], src_dict: Mapping[str, Any]) -> T:
        from ..models.upload_description_upload_request_headers import UploadDescriptionUploadRequestHeaders

        d = dict(src_dict)
        id = UUID(d.pop("id"))

        file_name = d.pop("fileName")

        mime_type = d.pop("mimeType")

        created_by = d.pop("createdBy")

        put_url = d.pop("putUrl")

        put_url_valid_until = isoparse(d.pop("putUrlValidUntil"))

        upload_request_headers = UploadDescriptionUploadRequestHeaders.from_dict(d.pop("uploadRequestHeaders"))

        upload_description = cls(
            id=id,
            file_name=file_name,
            mime_type=mime_type,
            created_by=created_by,
            put_url=put_url,
            put_url_valid_until=put_url_valid_until,
            upload_request_headers=upload_request_headers,
        )

        upload_description.additional_properties = d
        return upload_description

    @property
    def additional_keys(self) -> list[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties
