import datetime
from collections.abc import Mapping
from typing import Any, TypeVar
from uuid import UUID

from attrs import define as _attrs_define
from attrs import field as _attrs_field
from dateutil.parser import isoparse

from ..models.invitation_status import InvitationStatus
from ..models.invitation_type import InvitationType

T = TypeVar("T", bound="Invitation")


@_attrs_define
class Invitation:
    """
    Attributes:
        invitation_type (InvitationType):  Example: ADD.
        valid_until_date (datetime.date):  Example: 2028-05-30.
        id (UUID):  Example: de7f0e04-74fb-4657-9039-03dee0e6049b.
        invitation_status (InvitationStatus):  Example: ACTIVE.
        token (str):
    """

    invitation_type: InvitationType
    valid_until_date: datetime.date
    id: UUID
    invitation_status: InvitationStatus
    token: str
    additional_properties: dict[str, Any] = _attrs_field(init=False, factory=dict)

    def to_dict(self) -> dict[str, Any]:
        invitation_type = self.invitation_type.value

        valid_until_date = self.valid_until_date.isoformat()

        id = str(self.id)

        invitation_status = self.invitation_status.value

        token = self.token

        field_dict: dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "invitationType": invitation_type,
                "validUntilDate": valid_until_date,
                "id": id,
                "invitationStatus": invitation_status,
                "token": token,
            }
        )

        return field_dict

    @classmethod
    def from_dict(cls: type[T], src_dict: Mapping[str, Any]) -> T:
        d = dict(src_dict)
        invitation_type = InvitationType(d.pop("invitationType"))

        valid_until_date = isoparse(d.pop("validUntilDate")).date()

        id = UUID(d.pop("id"))

        invitation_status = InvitationStatus(d.pop("invitationStatus"))

        token = d.pop("token")

        invitation = cls(
            invitation_type=invitation_type,
            valid_until_date=valid_until_date,
            id=id,
            invitation_status=invitation_status,
            token=token,
        )

        invitation.additional_properties = d
        return invitation

    @property
    def additional_keys(self) -> list[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties
