from http import HTTPStatus
from typing import Any, Optional, Union, cast
from uuid import UUID

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.energy_production import EnergyProduction
from ...types import UNSET, Response, Unset


def _get_kwargs(
    user_ctx_id: UUID,
    building_id: UUID,
    room_id: UUID,
    inventory_id: UUID,
    *,
    user_language: Union[Unset, str] = UNSET,
) -> dict[str, Any]:
    headers: dict[str, Any] = {}
    if not isinstance(user_language, Unset):
        headers["User-Language"] = user_language

    _kwargs: dict[str, Any] = {
        "method": "get",
        "url": f"/reod/v1/{user_ctx_id}/building/{building_id}/room/{room_id}/inventory/{inventory_id}/energy-production",
    }

    _kwargs["headers"] = headers
    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Any, EnergyProduction]]:
    if response.status_code == 200:
        response_200 = EnergyProduction.from_dict(response.json())

        return response_200
    if response.status_code == 400:
        response_400 = cast(Any, None)
        return response_400
    if response.status_code == 401:
        response_401 = cast(Any, None)
        return response_401
    if response.status_code == 403:
        response_403 = cast(Any, None)
        return response_403
    if response.status_code == 404:
        response_404 = cast(Any, None)
        return response_404
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Any, EnergyProduction]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    user_ctx_id: UUID,
    building_id: UUID,
    room_id: UUID,
    inventory_id: UUID,
    *,
    client: Union[AuthenticatedClient, Client],
    user_language: Union[Unset, str] = UNSET,
) -> Response[Union[Any, EnergyProduction]]:
    """Get EnergyProduction

     Return the EnergyProduction of an Installation.

    Args:
        user_ctx_id (UUID):
        building_id (UUID):
        room_id (UUID):
        inventory_id (UUID):
        user_language (Union[Unset, str]):  Example: ['deu'].

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, EnergyProduction]]
    """

    kwargs = _get_kwargs(
        user_ctx_id=user_ctx_id,
        building_id=building_id,
        room_id=room_id,
        inventory_id=inventory_id,
        user_language=user_language,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    user_ctx_id: UUID,
    building_id: UUID,
    room_id: UUID,
    inventory_id: UUID,
    *,
    client: Union[AuthenticatedClient, Client],
    user_language: Union[Unset, str] = UNSET,
) -> Optional[Union[Any, EnergyProduction]]:
    """Get EnergyProduction

     Return the EnergyProduction of an Installation.

    Args:
        user_ctx_id (UUID):
        building_id (UUID):
        room_id (UUID):
        inventory_id (UUID):
        user_language (Union[Unset, str]):  Example: ['deu'].

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, EnergyProduction]
    """

    return sync_detailed(
        user_ctx_id=user_ctx_id,
        building_id=building_id,
        room_id=room_id,
        inventory_id=inventory_id,
        client=client,
        user_language=user_language,
    ).parsed


async def asyncio_detailed(
    user_ctx_id: UUID,
    building_id: UUID,
    room_id: UUID,
    inventory_id: UUID,
    *,
    client: Union[AuthenticatedClient, Client],
    user_language: Union[Unset, str] = UNSET,
) -> Response[Union[Any, EnergyProduction]]:
    """Get EnergyProduction

     Return the EnergyProduction of an Installation.

    Args:
        user_ctx_id (UUID):
        building_id (UUID):
        room_id (UUID):
        inventory_id (UUID):
        user_language (Union[Unset, str]):  Example: ['deu'].

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, EnergyProduction]]
    """

    kwargs = _get_kwargs(
        user_ctx_id=user_ctx_id,
        building_id=building_id,
        room_id=room_id,
        inventory_id=inventory_id,
        user_language=user_language,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    user_ctx_id: UUID,
    building_id: UUID,
    room_id: UUID,
    inventory_id: UUID,
    *,
    client: Union[AuthenticatedClient, Client],
    user_language: Union[Unset, str] = UNSET,
) -> Optional[Union[Any, EnergyProduction]]:
    """Get EnergyProduction

     Return the EnergyProduction of an Installation.

    Args:
        user_ctx_id (UUID):
        building_id (UUID):
        room_id (UUID):
        inventory_id (UUID):
        user_language (Union[Unset, str]):  Example: ['deu'].

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, EnergyProduction]
    """

    return (
        await asyncio_detailed(
            user_ctx_id=user_ctx_id,
            building_id=building_id,
            room_id=room_id,
            inventory_id=inventory_id,
            client=client,
            user_language=user_language,
        )
    ).parsed
