from http import HTTPStatus
from typing import Any, Optional, Union, cast
from uuid import UUID

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.building import Building
from ...types import UNSET, Response, Unset


def _get_kwargs(
    user_ctx_id: UUID,
    *,
    body: Building,
    user_language: Union[Unset, str] = UNSET,
) -> dict[str, Any]:
    headers: dict[str, Any] = {}
    if not isinstance(user_language, Unset):
        headers["User-Language"] = user_language

    _kwargs: dict[str, Any] = {
        "method": "post",
        "url": f"/reod/v1/{user_ctx_id}/building",
    }

    _body = body.to_dict()

    _kwargs["json"] = _body
    headers["Content-Type"] = "application/json"

    _kwargs["headers"] = headers
    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Any, Building]]:
    if response.status_code == 201:
        response_201 = Building.from_dict(response.json())

        return response_201
    if response.status_code == 400:
        response_400 = cast(Any, None)
        return response_400
    if response.status_code == 401:
        response_401 = cast(Any, None)
        return response_401
    if response.status_code == 403:
        response_403 = cast(Any, None)
        return response_403
    if response.status_code == 404:
        response_404 = cast(Any, None)
        return response_404
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Any, Building]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    user_ctx_id: UUID,
    *,
    client: Union[AuthenticatedClient, Client],
    body: Building,
    user_language: Union[Unset, str] = UNSET,
) -> Response[Union[Any, Building]]:
    """Add Building

     Creates a new Building.

    Args:
        user_ctx_id (UUID):
        user_language (Union[Unset, str]):  Example: ['deu'].
        body (Building):  Example: [{'id': '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'egid':
            'string', 'plotId': 'ef535cb1-61ae-4034-af79-9bdd6b359726', 'buildingType': {'id':
            '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code': 'string', 'name': 'string', 'description':
            'string', 'groupMemberOf': [{'id': '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code':
            'string', 'name': 'string', 'description': 'string'}]}, 'buildingClass': {'id':
            '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code': 'string', 'name': 'string', 'description':
            'string', 'groupMemberOf': [{'id': '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code':
            'string', 'name': 'string', 'description': 'string'}]}, 'heightInMeters': 0,
            'groundAreaInM2': 0, 'livingAreaInM2': 0, 'cubicVolumeInM3': 0, 'numberOfapartments': 0,
            'numberOfFloors': 0, 'roofAreaInM2': 0, 'facadeAreaInM2': 0}].

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, Building]]
    """

    kwargs = _get_kwargs(
        user_ctx_id=user_ctx_id,
        body=body,
        user_language=user_language,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    user_ctx_id: UUID,
    *,
    client: Union[AuthenticatedClient, Client],
    body: Building,
    user_language: Union[Unset, str] = UNSET,
) -> Optional[Union[Any, Building]]:
    """Add Building

     Creates a new Building.

    Args:
        user_ctx_id (UUID):
        user_language (Union[Unset, str]):  Example: ['deu'].
        body (Building):  Example: [{'id': '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'egid':
            'string', 'plotId': 'ef535cb1-61ae-4034-af79-9bdd6b359726', 'buildingType': {'id':
            '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code': 'string', 'name': 'string', 'description':
            'string', 'groupMemberOf': [{'id': '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code':
            'string', 'name': 'string', 'description': 'string'}]}, 'buildingClass': {'id':
            '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code': 'string', 'name': 'string', 'description':
            'string', 'groupMemberOf': [{'id': '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code':
            'string', 'name': 'string', 'description': 'string'}]}, 'heightInMeters': 0,
            'groundAreaInM2': 0, 'livingAreaInM2': 0, 'cubicVolumeInM3': 0, 'numberOfapartments': 0,
            'numberOfFloors': 0, 'roofAreaInM2': 0, 'facadeAreaInM2': 0}].

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, Building]
    """

    return sync_detailed(
        user_ctx_id=user_ctx_id,
        client=client,
        body=body,
        user_language=user_language,
    ).parsed


async def asyncio_detailed(
    user_ctx_id: UUID,
    *,
    client: Union[AuthenticatedClient, Client],
    body: Building,
    user_language: Union[Unset, str] = UNSET,
) -> Response[Union[Any, Building]]:
    """Add Building

     Creates a new Building.

    Args:
        user_ctx_id (UUID):
        user_language (Union[Unset, str]):  Example: ['deu'].
        body (Building):  Example: [{'id': '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'egid':
            'string', 'plotId': 'ef535cb1-61ae-4034-af79-9bdd6b359726', 'buildingType': {'id':
            '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code': 'string', 'name': 'string', 'description':
            'string', 'groupMemberOf': [{'id': '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code':
            'string', 'name': 'string', 'description': 'string'}]}, 'buildingClass': {'id':
            '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code': 'string', 'name': 'string', 'description':
            'string', 'groupMemberOf': [{'id': '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code':
            'string', 'name': 'string', 'description': 'string'}]}, 'heightInMeters': 0,
            'groundAreaInM2': 0, 'livingAreaInM2': 0, 'cubicVolumeInM3': 0, 'numberOfapartments': 0,
            'numberOfFloors': 0, 'roofAreaInM2': 0, 'facadeAreaInM2': 0}].

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, Building]]
    """

    kwargs = _get_kwargs(
        user_ctx_id=user_ctx_id,
        body=body,
        user_language=user_language,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    user_ctx_id: UUID,
    *,
    client: Union[AuthenticatedClient, Client],
    body: Building,
    user_language: Union[Unset, str] = UNSET,
) -> Optional[Union[Any, Building]]:
    """Add Building

     Creates a new Building.

    Args:
        user_ctx_id (UUID):
        user_language (Union[Unset, str]):  Example: ['deu'].
        body (Building):  Example: [{'id': '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'egid':
            'string', 'plotId': 'ef535cb1-61ae-4034-af79-9bdd6b359726', 'buildingType': {'id':
            '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code': 'string', 'name': 'string', 'description':
            'string', 'groupMemberOf': [{'id': '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code':
            'string', 'name': 'string', 'description': 'string'}]}, 'buildingClass': {'id':
            '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code': 'string', 'name': 'string', 'description':
            'string', 'groupMemberOf': [{'id': '497f6eca-6276-4993-bfeb-53cbbbba6f08', 'code':
            'string', 'name': 'string', 'description': 'string'}]}, 'heightInMeters': 0,
            'groundAreaInM2': 0, 'livingAreaInM2': 0, 'cubicVolumeInM3': 0, 'numberOfapartments': 0,
            'numberOfFloors': 0, 'roofAreaInM2': 0, 'facadeAreaInM2': 0}].

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, Building]
    """

    return (
        await asyncio_detailed(
            user_ctx_id=user_ctx_id,
            client=client,
            body=body,
            user_language=user_language,
        )
    ).parsed
